﻿//----------------------------------------------------------
// Copyright (C) Microsoft Corporation. All rights reserved.
//----------------------------------------------------------
// TFS.UI.js
//
/// <reference path="~/Scripts/DevTime.js" />
/// <reference path=~/Scripts/jquery-1.6.2-vsdoc.js" />
/// <reference path="~/Scripts/MicrosoftAjax-4.0.0.0.debug.js"/>
/// <reference path="~/Scripts/TFS/TFS.debug.js" />
/// <reference path="~/Scripts/TFS/TFS.Core.debug.js" />
/// <reference path="~/Scripts/TFS/TFS.Diag.debug.js" />


TFS.module("TFS.UI", ["TFS.Core", "TFS.Diag"], function () {
    var log = TFS.Diag.log,
        Diag = TFS.Diag,
        verbose = TFS.Diag.LogVerbosity.Verbose,
        delegate = TFS.Core.delegate;




    function getWheelDelta(e) {
        var result = 0;
        if (e.wheelDelta) {
            result = event.wheelDelta / 120;
        }
        else if (e.detail) { // firefox gives delta differently
            result = -e.detail / 3;
        }
        return result;
    }

    function enableElement(element, enable) {
        if (enable === true) {
            $(element).removeAttr("disabled");
        }
        else {
            $(element).attr("disabled", "disabled");
        }
    }

    function addClass(element, className, add) {
        if (add === true) {
            $(element).addClass(className);
        }
        else {
            $(element).removeClass(className);
        }
    }

    function makeElementUnselectable(element) {
        if ($.browser.mozilla) {
            element.style.MozUserFocus = "ignore";
        }
        else if ($.browser.msie) {
            element.unselectable = "on";

            var e, i = 0;
            while (e = element.all[i++]) {
                switch (e.tagName) {
                    case "IFRAME":
                    case "TEXTAREA":
                    case "INPUT":
                    case "SELECT":
                        break;
                    default:
                        e.unselectable = "on";
                }
            }
        }
    }

    function tryFocus(element, delay) {
        /// <summary>Best-effort attempt to set focus on the specified element. Exceptions will be caught and logged to console.</summary>
        /// <param name="element" type="DomElement or jQuery element">Element to set focus on</param>
        /// <param name="delay" type="int">Optional delay in ms before calling focus</param>

        var $element = element instanceof jQuery ? element : $(element);

        function doSetFocus() {
            try {
                $element.focus();
            }
            catch (ex) {
                log(TFS.Diag.LogVerbosity.Warning, "Failed to set focus to element: " + ex);
            }
        }

        if (typeof delay === "undefined") {
            doSetFocus();
        }
        else {
            TFS.Core.delay(this, delay, function () {
                doSetFocus();
            });
        }
    }

    function alignWidth(element, baseWidth) {
        var $element = $(element);
        var widthFix = $element.outerWidth() - baseWidth;
        if (widthFix != 0) {
            // Fixing the width
            $element.width(baseWidth - widthFix);
        }
    }

    function isInDomTree(element) {
        return $(element).parent().length > 0;
    }

    function getCustomData(element, key) {
        if (!element.dataset) {
            return element.getAttribute("data-" + key);
        }
        else {
            return element.dataset[key];
        }
    }

    function KeyCode() {
    }

    KeyCode.extend({
        ALT: 18,
        BACKSPACE: 8,
        CAPS_LOCK: 20,
        COMMA: 188,
        CONTROL: 17,
        DELETE: 46,
        DOWN: 40,
        END: 35,
        ENTER: 13,
        ESCAPE: 27,
        HOME: 36,
        INSERT: 45,
        LEFT: 37,
        PAGE_DOWN: 34,
        PAGE_UP: 33,
        PERIOD: 190,
        RIGHT: 39,
        SHIFT: 16,
        SPACE: 32,
        TAB: 9,
        UP: 38,
        F1: 112
    });

    function Constants() {
    }

    Constants.extend(function () {
        var blurTimeout = $.browser.msie ? 100 : 250;

        return {
            HtmlNewLine: "<BR>",
            BlurTimeout: blurTimeout
        };
    }());

    function Measurement() {
    }

    Measurement.extend(function () {
        var id = "tfs-measurement-probe";

        function createProbe($parent) {
            // Creating Measurement div for em and ex
            var $probe = $("<div/>")
            .attr("id", id)
            .css("position", "absolute")
            .css("left", "-50000px")
            .css("top", "-50000px")
            .css("width", "9em")
            .css("height", "18ex")
            .css("overflow", "auto")
            .appendTo($parent);

            // Creating Measurement div for in and scrollbars
            $("<div></div>")
            .css("width", "3in")
            .css("height", "36ex")
            .appendTo($probe);

            return $probe;
        }

        function getProbe($parent) {
            // If no parent specified, adding the probe to document.body
            $parent = $parent || $(document.body);
            var $probe = $parent.children("#" + id);

            // Checking the probe already exists. If not exists creating one
            if ($probe.length === 0) {
                $probe = createProbe($parent);
            }

            return $probe;
        }

        function getUnitEm($parent) {
            return getProbe($parent).width() / 9;
        }

        function getUnitEx($parent) {
            return getProbe($parent).height() / 18;
        }

        function getUnitIn($parent) {
            return $("div", getProbe($parent)).width() / 3;
        }

        function getScrollbarWidth($parent) {
            var $probe = getProbe($parent);
            return $probe.width() - $probe[0].clientWidth;
        }

        function getScrollbarHeight($parent) {
            var $probe = getProbe($parent);
            return $probe.height() - $probe[0].clientHeight;
        }

        function getUnitAsPixel(unit, $parent) {
            unit = unit.toLowerCase();
            var result = parseInt(unit);

            if (unit.indexOf("px") > 0) {
                result = Math.round(parseFloat(unit));
            }
            else if (unit.indexOf("em") > 0) {
                result = Math.round(parseFloat(unit) * getUnitEm($parent));
            }
            else if (unit.indexOf("ex") > 0) {
                result = Math.round(parseFloat(unit) * getUnitEx($parent));
            }
            else if (unit.indexOf("in") > 0) {
                result = Math.round(parseFloat(unit) * getUnitIn($parent));
            }
            else if (unit.indexOf("mm") > 0) {
                result = Math.round(parseFloat(unit) * getUnitIn($parent) / 25.4);
            }
            else if (unit.indexOf("cm") > 0) {
                result = Math.round(parseFloat(unit) * getUnitIn($parent) / 2.54);
            }
            return result;
        }

        return {
            getUnitEm: getUnitEm,
            getUnitEx: getUnitEx,
            getUnitIn: getUnitIn,
            getScrollbarWidth: getScrollbarWidth,
            getScrollbarHeight: getScrollbarHeight,
            getUnitAsPixel: getUnitAsPixel
        };
    }());

    function domElem(tagName, className) {
        var element = document.createElement(tagName);

        if (className) {
            element.className = className;
        }

        return element;
    }

    function ensure$(element) {
        return element.jquery ? element : $(element);
    }

    function htmlEncode(input) {
        if (!input) {
            return "";
        }

        var result = input;

        if (typeof input !== "string") {
            result = "" + input;
        }

        return result.replace(/&/g, "&amp;").replace(/>/g, "&gt;").replace(/</g, "&lt;");
    }

    function Positioning() {
    }

    Positioning.extend({
        _topOverflow: function (top) {
            return (0 - top);
        },

        _bottomOverflow: function (bottom) {
            var $window = $(window),
                windowHeight = $window.height() + $window.scrollTop();

            return (bottom - windowHeight);
        },

        _fitHorizontal: function (position, data) {
            var $window = $(window),
                over = position.left + data.elementMeasure - ($window.width() + $window.scrollLeft());

            position.left = over > 0 ? position.left - over : Math.max(0, position.left);
        },
        _flipHorizontal: function (position, data) {
            var $window = $(window),
                over = position.left + data.elementMeasure - ($window.width() + $window.scrollLeft()),
                offset;

            offset = data.elementAlign === "left" ? -data.elementMeasure : data.elementMeasure;

            position.left += position.left < 0
                ? offset + data.baseMeasure
                : (over > 0
                    ? offset - data.baseMeasure
                    : 0);
        },

        _fitVertical: function (position, data) {
            /// <summary>Tries to fit the positioned element by using the base element if any overflow exists.
            /// If still overflow exists after flipping, it shrinks the element where it best fits.</summary>

            var newTop, shrink = 0,
                topOverflow, bottomOverflow;

            newTop = position.top;

            // Checking top overflow
            topOverflow = Positioning._topOverflow(newTop);
            if (topOverflow > 0) {
                // Top overflow exists, fitting the element
                newTop = 0;

                // Checking any overflow from bottom as a result of fit
                bottomOverflow = Positioning._bottomOverflow(newTop + data.elementMeasure);
                if (bottomOverflow > 0) {
                    // Bottom overflow exists. Shrinking the element to fit the screen.
                    shrink = bottomOverflow;
                }
            }
            else {
                // Checking bottom overflow
                bottomOverflow = Positioning._bottomOverflow(newTop + data.elementMeasure);
                if (bottomOverflow > 0) {
                    // Bottom overflow exists, fitting the element
                    newTop -= bottomOverflow;

                    // Checking any overflow from top as a result of fit
                    topOverflow = Positioning._topOverflow(newTop);
                    if (topOverflow > 0) {
                        // Top overflow exists. Shrinking the element to fit the screen.
                        newTop = 0;
                        shrink = topOverflow;
                    }
                }
            }

            return { top: newTop, shrink: shrink };
        },

        _flipVertical: function (position, data) {
            /// <summary>Tries to flip the positioned element by using the base element if any overflow exists.
            /// If still overflow exists after flipping, it shrinks the element where it best fits.</summary>

            var newTop, shrink,
                offset, baseOffset,
                topOverflow, bottomOverflow;

            newTop = position.top;
            offset = data.elementAlign === "top" ? -data.elementMeasure : data.elementMeasure;

            // Checking top overflow
            topOverflow = Positioning._topOverflow(newTop);
            if (topOverflow > 0) {
                // Top overflow exists, flipping the element
                newTop += (offset + data.baseMeasure);

                // Checking any overflow from bottom as a result of flip
                bottomOverflow = Positioning._bottomOverflow(newTop + data.elementMeasure);
                if (bottomOverflow > 0) {
                    // Bottom overflow exists. Trying to position the element in the area
                    // where fewest overflow occurs.
                    if (bottomOverflow >= topOverflow) {
                        // First position was better, recovering.
                        newTop = position.top;
                        shrink = topOverflow;
                    }
                    else {
                        // New position is better
                        shrink = bottomOverflow;
                    }
                }
            }
            else {
                // Checking bottom overflow
                bottomOverflow = Positioning._bottomOverflow(newTop + data.elementMeasure);
                if (bottomOverflow > 0) {

                    baseOffset = data.baseAlign === "top" ? data.baseMeasure : -data.baseMeasure;

                    // Bottom overflow exists, flipping the element
                    newTop += (offset + baseOffset);

                    // Checking any overflow from top as a result of flip
                    topOverflow = Positioning._topOverflow(newTop);
                    if (topOverflow > 0) {
                        // Top overflow exists. Trying to position the element in the area
                        // where fewest overflow occurs.
                        if (topOverflow >= bottomOverflow) {
                            // First position was better, recovering.
                            newTop = position.top;
                            shrink = bottomOverflow;
                        }
                        else {
                            // New position is better
                            newTop = 0;
                            shrink = topOverflow;
                        }
                    }
                }
            }

            return { top: newTop, shrink: shrink };
        },
        _getZoomFactor: function () {
            // Fixing the issue for the situations, where css returns actual values
            // for left and top when zoomed in or out.
            var probeLeft = -100000,
                        probeElement,
                        probeCssLeft,
                        probeOffsetLeft,
                        zoomFactor;

            // Creating an element to check the zoom level
            probeElement = jQuery("<div style='left: " + probeLeft + "px;width:1px;height:1px;position:absolute'/>")
                        .appendTo(document.body);

            // If the zoom level is 150%, Chrome, Safari and IE9 will return -150000
            probeCssLeft = parseInt(jQuery.curCSS(probeElement[0], "left", true), 10) || 1;
            // Offset function always returns the correct value
            probeOffsetLeft = probeElement.offset().left || 1;
            // Calculating the zoom factor
            zoomFactor = probeCssLeft / probeOffsetLeft;

            // Removing the probe element. We don't need it anymore.
            probeElement.remove();

            return zoomFactor;
        },
        position: function (element, baseElement, options) {
            /// <summary>Positions the given element by taking the given base element
            /// as a reference using the options provided</summary>
            /// <param name="element" type="DomElement">Element to position</param>
            /// <param name="baseElement" type="DomElement">Reference element for positioning</param>
            /// <param name="options" type="Object">The following options are supported:
            ///
            ///     - elementAlign: where to align the element (horizontal-vertical)
            ///     - baseAlign: where to align the element against base element (horizontal-vertical)
            ///     - overflow: behavior when the element overflows the window (horizontal-vertical)
            ///
            ///     Example usage:
            ///     TFS.UI.position(element, baseElement, { elementAlign: "left-top", baseAlign: "left-bottom" });
            ///</param>

            options = $.extend({}, options);

            var x = 0,  // horizontal
                y = 1,  // vertical
                position, // calculated position
			    elementWidth, elementHeight,
                basePosition, baseWidth, baseHeight,
                elementAlign, baseAlign,
                overflow,
                supportScroll, // positioning should support scroll or not (if overflow occurs)
                oHandlerX,  // horizontal overflow handler
                oHandlerY, // vertical overflow handler
                oResultY, // vertical overflow result
                zoom; // zoom factor

            element = ensure$(element);
            baseElement = ensure$(baseElement);

            supportScroll = options.supportScroll === true,

            // Moving element to top
            element.css("z-index", 20000);

            // Resetting height
            if (supportScroll) {
                element[0].style.height = "";
            }

            elementWidth = element.outerWidth();
            elementHeight = element.outerHeight();
            basePosition = baseElement.offset();
            baseWidth = baseElement.outerWidth();
            baseHeight = baseElement.outerHeight();
            elementAlign = options.elementAlign || "left-top";
            baseAlign = options.baseAlign || "left-bottom";
            overflow = options.overflow || "fit-flip";

            // Splitting the base align into two. First is horizontal,
            // second is vertical value;
            baseAlign = baseAlign.split("-");
            if (baseAlign[x] === "right") {
                basePosition.left += baseWidth;
            }

            if (baseAlign[y] === "bottom") {
                basePosition.top += baseHeight;
            }

            // Creating a copy of the base position
            position = $.extend({}, basePosition);

            // Splitting the element align into two. First is horizontal,
            // second is vertical value;
            elementAlign = elementAlign.split("-");
            if (elementAlign[x] === "right") {
                position.left -= elementWidth;
            }

            if (elementAlign[y] === "bottom") {
                position.top -= elementHeight;
            }

            position.left = Math.round(position.left);
            position.top = Math.round(position.top);

            // Splitting overflow into two. First is horizontal,
            // second is vertical value;
            overflow = overflow.split("-");

            // Fixing vertical overflow
            oHandlerY = Positioning["_" + overflow[y].toLowerCase() + "Vertical"];
            if (oHandlerY) {
                oResultY = oHandlerY(position, {
                    baseMeasure: baseHeight,
                    baseAlign: baseAlign[y],
                    elementMeasure: elementHeight,
                    elementAlign: elementAlign[y]
                });

                position.top = oResultY.top;

                if (supportScroll && oResultY.shrink > 0) {
                    element.height(element.height() - oResultY.shrink);

                    // We need to re-measure width because of the scrollbar
                    elementWidth = element.outerWidth();
                }
            }

            // Fixing horizontal overflow
            oHandlerX = Positioning["_" + overflow[x].toLowerCase() + "Horizontal"];
            if ($.isFunction(oHandlerX)) {
                oHandlerX(position, {
                    baseMeasure: baseWidth,
                    baseAlign: baseAlign[x],
                    elementMeasure: elementWidth,
                    elementAlign: elementAlign[x]
                });
            }

            var curOffset = element.offset(),
			curTop = parseInt(jQuery.curCSS(element[0], "top", true), 10) || 0,
			curLeft = parseInt(jQuery.curCSS(element[0], "left", true), 10) || 0;

            zoom = Positioning._getZoomFactor();

            // If the page is zoomed in or out
            if (zoom !== 1) {
                // Correcting top and left which have actual values
                curTop = curTop / zoom;
                curLeft = curLeft / zoom;
            }

            var props = {
                top: (position.top - curOffset.top) + curTop,
                left: (position.left - curOffset.left) + curLeft
            };

            // Setting the position
            element.css(props);
        }
    });


    return {
        getWheelDelta: getWheelDelta,
        enableElement: enableElement,
        addClass: addClass,
        alignWidth: alignWidth,
        makeElementUnselectable: makeElementUnselectable,
        tryFocus: tryFocus,
        isInDomTree: isInDomTree,
        getCustomData: getCustomData,
        KeyCode: KeyCode,
        Measurement: Measurement,
        Constants: Constants,
        domElem: domElem,
        htmlEncode: htmlEncode,
        position: Positioning.position
    };
});

// SIG // Begin signature block
// SIG // MIIapwYJKoZIhvcNAQcCoIIamDCCGpQCAQExCzAJBgUr
// SIG // DgMCGgUAMGcGCisGAQQBgjcCAQSgWTBXMDIGCisGAQQB
// SIG // gjcCAR4wJAIBAQQQEODJBs441BGiowAQS9NQkAIBAAIB
// SIG // AAIBAAIBAAIBADAhMAkGBSsOAwIaBQAEFETKgvHgA8fP
// SIG // D2PawvRWC0fgAK2BoIIVgjCCBMMwggOroAMCAQICEzMA
// SIG // AAArOTJIwbLJSPMAAAAAACswDQYJKoZIhvcNAQEFBQAw
// SIG // dzELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0
// SIG // b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1p
// SIG // Y3Jvc29mdCBDb3Jwb3JhdGlvbjEhMB8GA1UEAxMYTWlj
// SIG // cm9zb2Z0IFRpbWUtU3RhbXAgUENBMB4XDTEyMDkwNDIx
// SIG // MTIzNFoXDTEzMTIwNDIxMTIzNFowgbMxCzAJBgNVBAYT
// SIG // AlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQH
// SIG // EwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29y
// SIG // cG9yYXRpb24xDTALBgNVBAsTBE1PUFIxJzAlBgNVBAsT
// SIG // Hm5DaXBoZXIgRFNFIEVTTjpDMEY0LTMwODYtREVGODEl
// SIG // MCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2Vy
// SIG // dmljZTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoC
// SIG // ggEBAKa2MA4DZa5QWoZrhZ9IoR7JwO5eSQeF4HCWfL65
// SIG // X2JfBibTizm7GCKlLpKt2EuIOhqvm4OuyF45jMIyexZ4
// SIG // 7Tc4OvFi+2iCAmjs67tAirH+oSw2YmBwOWBiDvvGGDhv
// SIG // sJLWQA2Apg14izZrhoomFxj/sOtNurspE+ZcSI5wRjYm
// SIG // /jQ1qzTh99rYXOqZfTG3TR9X63zWlQ1mDB4OMhc+LNWA
// SIG // oc7r95iRAtzBX/04gPg5f11kyjdcO1FbXYVfzh4c+zS+
// SIG // X+UoVXBUnLjsfABVRlsomChWTOHxugkZloFIKjDI9zMg
// SIG // bOdpw7PUw07PMB431JhS1KkjRbKuXEFJT7RiaJMCAwEA
// SIG // AaOCAQkwggEFMB0GA1UdDgQWBBSlGDNTP5VgoUMW747G
// SIG // r9Irup5Y0DAfBgNVHSMEGDAWgBQjNPjZUkZwCu1A+3b7
// SIG // syuwwzWzDzBUBgNVHR8ETTBLMEmgR6BFhkNodHRwOi8v
// SIG // Y3JsLm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0
// SIG // cy9NaWNyb3NvZnRUaW1lU3RhbXBQQ0EuY3JsMFgGCCsG
// SIG // AQUFBwEBBEwwSjBIBggrBgEFBQcwAoY8aHR0cDovL3d3
// SIG // dy5taWNyb3NvZnQuY29tL3BraS9jZXJ0cy9NaWNyb3Nv
// SIG // ZnRUaW1lU3RhbXBQQ0EuY3J0MBMGA1UdJQQMMAoGCCsG
// SIG // AQUFBwMIMA0GCSqGSIb3DQEBBQUAA4IBAQB+zLB75S++
// SIG // 51a1z3PbqlLRFjnGtM361/4eZbXnSPObRogFZmomhl7+
// SIG // h1jcxmOOOID0CEZ8K3OxDr9BqsvHqpSkN/BkOeHF1fnO
// SIG // B86r5CXwaa7URuL+ZjI815fFMiH67holoF4MQiwRMzqC
// SIG // g/3tHbO+zpGkkSVxuatysJ6v5M8AYolwqbhKUIzuLyJk
// SIG // pajmTWuVLBx57KejMdqQYJCkbv6TAg0/LCQNxmomgVGD
// SIG // ShC7dWNEqmkIxgPr4s8L7VY67O9ypwoM9ADTIrivInKz
// SIG // 58ScCyiggMrj4dc5ZjDnRhcY5/qC+lkLeryoDf4c/wOL
// SIG // Y7JNEgIjTy2zhYQ74qFH6M8VMIIE7DCCA9SgAwIBAgIT
// SIG // MwAAALARrwqL0Duf3QABAAAAsDANBgkqhkiG9w0BAQUF
// SIG // ADB5MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGlu
// SIG // Z3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMV
// SIG // TWljcm9zb2Z0IENvcnBvcmF0aW9uMSMwIQYDVQQDExpN
// SIG // aWNyb3NvZnQgQ29kZSBTaWduaW5nIFBDQTAeFw0xMzAx
// SIG // MjQyMjMzMzlaFw0xNDA0MjQyMjMzMzlaMIGDMQswCQYD
// SIG // VQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4G
// SIG // A1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
// SIG // IENvcnBvcmF0aW9uMQ0wCwYDVQQLEwRNT1BSMR4wHAYD
// SIG // VQQDExVNaWNyb3NvZnQgQ29ycG9yYXRpb24wggEiMA0G
// SIG // CSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDor1yiIA34
// SIG // KHy8BXt/re7rdqwoUz8620B9s44z5lc/pVEVNFSlz7SL
// SIG // qT+oN+EtUO01Fk7vTXrbE3aIsCzwWVyp6+HXKXXkG4Un
// SIG // m/P4LZ5BNisLQPu+O7q5XHWTFlJLyjPFN7Dz636o9UEV
// SIG // XAhlHSE38Cy6IgsQsRCddyKFhHxPuRuQsPWj/ov0DJpO
// SIG // oPXJCiHiquMBNkf9L4JqgQP1qTXclFed+0vUDoLbOI8S
// SIG // /uPWenSIZOFixCUuKq6dGB8OHrbCryS0DlC83hyTXEmm
// SIG // ebW22875cHsoAYS4KinPv6kFBeHgD3FN/a1cI4Mp68fF
// SIG // SsjoJ4TTfsZDC5UABbFPZXHFAgMBAAGjggFgMIIBXDAT
// SIG // BgNVHSUEDDAKBggrBgEFBQcDAzAdBgNVHQ4EFgQUWXGm
// SIG // WjNN2pgHgP+EHr6H+XIyQfIwUQYDVR0RBEowSKRGMEQx
// SIG // DTALBgNVBAsTBE1PUFIxMzAxBgNVBAUTKjMxNTk1KzRm
// SIG // YWYwYjcxLWFkMzctNGFhMy1hNjcxLTc2YmMwNTIzNDRh
// SIG // ZDAfBgNVHSMEGDAWgBTLEejK0rQWWAHJNy4zFha5TJoK
// SIG // HzBWBgNVHR8ETzBNMEugSaBHhkVodHRwOi8vY3JsLm1p
// SIG // Y3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWND
// SIG // b2RTaWdQQ0FfMDgtMzEtMjAxMC5jcmwwWgYIKwYBBQUH
// SIG // AQEETjBMMEoGCCsGAQUFBzAChj5odHRwOi8vd3d3Lm1p
// SIG // Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY0NvZFNpZ1BD
// SIG // QV8wOC0zMS0yMDEwLmNydDANBgkqhkiG9w0BAQUFAAOC
// SIG // AQEAMdduKhJXM4HVncbr+TrURE0Inu5e32pbt3nPApy8
// SIG // dmiekKGcC8N/oozxTbqVOfsN4OGb9F0kDxuNiBU6fNut
// SIG // zrPJbLo5LEV9JBFUJjANDf9H6gMH5eRmXSx7nR2pEPoc
// SIG // sHTyT2lrnqkkhNrtlqDfc6TvahqsS2Ke8XzAFH9IzU2y
// SIG // RPnwPJNtQtjofOYXoJtoaAko+QKX7xEDumdSrcHps3Om
// SIG // 0mPNSuI+5PNO/f+h4LsCEztdIN5VP6OukEAxOHUoXgSp
// SIG // Rm3m9Xp5QL0fzehF1a7iXT71dcfmZmNgzNWahIeNJDD3
// SIG // 7zTQYx2xQmdKDku/Og7vtpU6pzjkJZIIpohmgjCCBbww
// SIG // ggOkoAMCAQICCmEzJhoAAAAAADEwDQYJKoZIhvcNAQEF
// SIG // BQAwXzETMBEGCgmSJomT8ixkARkWA2NvbTEZMBcGCgmS
// SIG // JomT8ixkARkWCW1pY3Jvc29mdDEtMCsGA1UEAxMkTWlj
// SIG // cm9zb2Z0IFJvb3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5
// SIG // MB4XDTEwMDgzMTIyMTkzMloXDTIwMDgzMTIyMjkzMlow
// SIG // eTELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0
// SIG // b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1p
// SIG // Y3Jvc29mdCBDb3Jwb3JhdGlvbjEjMCEGA1UEAxMaTWlj
// SIG // cm9zb2Z0IENvZGUgU2lnbmluZyBQQ0EwggEiMA0GCSqG
// SIG // SIb3DQEBAQUAA4IBDwAwggEKAoIBAQCycllcGTBkvx2a
// SIG // YCAgQpl2U2w+G9ZvzMvx6mv+lxYQ4N86dIMaty+gMuz/
// SIG // 3sJCTiPVcgDbNVcKicquIEn08GisTUuNpb15S3GbRwfa
// SIG // /SXfnXWIz6pzRH/XgdvzvfI2pMlcRdyvrT3gKGiXGqel
// SIG // cnNW8ReU5P01lHKg1nZfHndFg4U4FtBzWwW6Z1KNpbJp
// SIG // L9oZC/6SdCnidi9U3RQwWfjSjWL9y8lfRjFQuScT5EAw
// SIG // z3IpECgixzdOPaAyPZDNoTgGhVxOVoIoKgUyt0vXT2Pn
// SIG // 0i1i8UU956wIAPZGoZ7RW4wmU+h6qkryRs83PDietHdc
// SIG // pReejcsRj1Y8wawJXwPTAgMBAAGjggFeMIIBWjAPBgNV
// SIG // HRMBAf8EBTADAQH/MB0GA1UdDgQWBBTLEejK0rQWWAHJ
// SIG // Ny4zFha5TJoKHzALBgNVHQ8EBAMCAYYwEgYJKwYBBAGC
// SIG // NxUBBAUCAwEAATAjBgkrBgEEAYI3FQIEFgQU/dExTtMm
// SIG // ipXhmGA7qDFvpjy82C0wGQYJKwYBBAGCNxQCBAweCgBT
// SIG // AHUAYgBDAEEwHwYDVR0jBBgwFoAUDqyCYEBWJ5flJRP8
// SIG // KuEKU5VZ5KQwUAYDVR0fBEkwRzBFoEOgQYY/aHR0cDov
// SIG // L2NybC5taWNyb3NvZnQuY29tL3BraS9jcmwvcHJvZHVj
// SIG // dHMvbWljcm9zb2Z0cm9vdGNlcnQuY3JsMFQGCCsGAQUF
// SIG // BwEBBEgwRjBEBggrBgEFBQcwAoY4aHR0cDovL3d3dy5t
// SIG // aWNyb3NvZnQuY29tL3BraS9jZXJ0cy9NaWNyb3NvZnRS
// SIG // b290Q2VydC5jcnQwDQYJKoZIhvcNAQEFBQADggIBAFk5
// SIG // Pn8mRq/rb0CxMrVq6w4vbqhJ9+tfde1MOy3XQ60L/svp
// SIG // LTGjI8x8UJiAIV2sPS9MuqKoVpzjcLu4tPh5tUly9z7q
// SIG // QX/K4QwXaculnCAt+gtQxFbNLeNK0rxw56gNogOlVuC4
// SIG // iktX8pVCnPHz7+7jhh80PLhWmvBTI4UqpIIck+KUBx3y
// SIG // 4k74jKHK6BOlkU7IG9KPcpUqcW2bGvgc8FPWZ8wi/1wd
// SIG // zaKMvSeyeWNWRKJRzfnpo1hW3ZsCRUQvX/TartSCMm78
// SIG // pJUT5Otp56miLL7IKxAOZY6Z2/Wi+hImCWU4lPF6H0q7
// SIG // 0eFW6NB4lhhcyTUWX92THUmOLb6tNEQc7hAVGgBd3TVb
// SIG // Ic6YxwnuhQ6MT20OE049fClInHLR82zKwexwo1eSV32U
// SIG // jaAbSANa98+jZwp0pTbtLS8XyOZyNxL0b7E8Z4L5UrKN
// SIG // MxZlHg6K3RDeZPRvzkbU0xfpecQEtNP7LN8fip6sCvsT
// SIG // J0Ct5PnhqX9GuwdgR2VgQE6wQuxO7bN2edgKNAltHIAx
// SIG // H+IOVN3lofvlRxCtZJj/UBYufL8FIXrilUEnacOTj5XJ
// SIG // jdibIa4NXJzwoq6GaIMMai27dmsAHZat8hZ79haDJLmI
// SIG // z2qoRzEvmtzjcT3XAH5iR9HOiMm4GPoOco3Boz2vAkBq
// SIG // /2mbluIQqBC0N1AI1sM9MIIGBzCCA++gAwIBAgIKYRZo
// SIG // NAAAAAAAHDANBgkqhkiG9w0BAQUFADBfMRMwEQYKCZIm
// SIG // iZPyLGQBGRYDY29tMRkwFwYKCZImiZPyLGQBGRYJbWlj
// SIG // cm9zb2Z0MS0wKwYDVQQDEyRNaWNyb3NvZnQgUm9vdCBD
// SIG // ZXJ0aWZpY2F0ZSBBdXRob3JpdHkwHhcNMDcwNDAzMTI1
// SIG // MzA5WhcNMjEwNDAzMTMwMzA5WjB3MQswCQYDVQQGEwJV
// SIG // UzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMH
// SIG // UmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBv
// SIG // cmF0aW9uMSEwHwYDVQQDExhNaWNyb3NvZnQgVGltZS1T
// SIG // dGFtcCBQQ0EwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAw
// SIG // ggEKAoIBAQCfoWyx39tIkip8ay4Z4b3i48WZUSNQrc7d
// SIG // GE4kD+7Rp9FMrXQwIBHrB9VUlRVJlBtCkq6YXDAm2gBr
// SIG // 6Hu97IkHD/cOBJjwicwfyzMkh53y9GccLPx754gd6udO
// SIG // o6HBI1PKjfpFzwnQXq/QsEIEovmmbJNn1yjcRlOwhtDl
// SIG // KEYuJ6yGT1VSDOQDLPtqkJAwbofzWTCd+n7Wl7PoIZd+
// SIG // +NIT8wi3U21StEWQn0gASkdmEScpZqiX5NMGgUqi+YSn
// SIG // EUcUCYKfhO1VeP4Bmh1QCIUAEDBG7bfeI0a7xC1Un68e
// SIG // eEExd8yb3zuDk6FhArUdDbH895uyAc4iS1T/+QXDwiAL
// SIG // AgMBAAGjggGrMIIBpzAPBgNVHRMBAf8EBTADAQH/MB0G
// SIG // A1UdDgQWBBQjNPjZUkZwCu1A+3b7syuwwzWzDzALBgNV
// SIG // HQ8EBAMCAYYwEAYJKwYBBAGCNxUBBAMCAQAwgZgGA1Ud
// SIG // IwSBkDCBjYAUDqyCYEBWJ5flJRP8KuEKU5VZ5KShY6Rh
// SIG // MF8xEzARBgoJkiaJk/IsZAEZFgNjb20xGTAXBgoJkiaJ
// SIG // k/IsZAEZFgltaWNyb3NvZnQxLTArBgNVBAMTJE1pY3Jv
// SIG // c29mdCBSb290IENlcnRpZmljYXRlIEF1dGhvcml0eYIQ
// SIG // ea0WoUqgpa1Mc1j0BxMuZTBQBgNVHR8ESTBHMEWgQ6BB
// SIG // hj9odHRwOi8vY3JsLm1pY3Jvc29mdC5jb20vcGtpL2Ny
// SIG // bC9wcm9kdWN0cy9taWNyb3NvZnRyb290Y2VydC5jcmww
// SIG // VAYIKwYBBQUHAQEESDBGMEQGCCsGAQUFBzAChjhodHRw
// SIG // Oi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpL2NlcnRzL01p
// SIG // Y3Jvc29mdFJvb3RDZXJ0LmNydDATBgNVHSUEDDAKBggr
// SIG // BgEFBQcDCDANBgkqhkiG9w0BAQUFAAOCAgEAEJeKw1wD
// SIG // RDbd6bStd9vOeVFNAbEudHFbbQwTq86+e4+4LtQSooxt
// SIG // YrhXAstOIBNQmd16QOJXu69YmhzhHQGGrLt48ovQ7DsB
// SIG // 7uK+jwoFyI1I4vBTFd1Pq5Lk541q1YDB5pTyBi+FA+mR
// SIG // KiQicPv2/OR4mS4N9wficLwYTp2OawpylbihOZxnLcVR
// SIG // DupiXD8WmIsgP+IHGjL5zDFKdjE9K3ILyOpwPf+FChPf
// SIG // wgphjvDXuBfrTot/xTUrXqO/67x9C0J71FNyIe4wyrt4
// SIG // ZVxbARcKFA7S2hSY9Ty5ZlizLS/n+YWGzFFW6J1wlGys
// SIG // OUzU9nm/qhh6YinvopspNAZ3GmLJPR5tH4LwC8csu89D
// SIG // s+X57H2146SodDW4TsVxIxImdgs8UoxxWkZDFLyzs7BN
// SIG // Z8ifQv+AeSGAnhUwZuhCEl4ayJ4iIdBD6Svpu/RIzCzU
// SIG // 2DKATCYqSCRfWupW76bemZ3KOm+9gSd0BhHudiG/m4LB
// SIG // J1S2sWo9iaF2YbRuoROmv6pH8BJv/YoybLL+31HIjCPJ
// SIG // Zr2dHYcSZAI9La9Zj7jkIeW1sMpjtHhUBdRBLlCslLCl
// SIG // eKuzoJZ1GtmShxN1Ii8yqAhuoFuMJb+g74TKIdbrHk/J
// SIG // mu5J4PcBZW+JC33Iacjmbuqnl84xKf8OxVtc2E0bodj6
// SIG // L54/LlUWa8kTo/0xggSRMIIEjQIBATCBkDB5MQswCQYD
// SIG // VQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4G
// SIG // A1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
// SIG // IENvcnBvcmF0aW9uMSMwIQYDVQQDExpNaWNyb3NvZnQg
// SIG // Q29kZSBTaWduaW5nIFBDQQITMwAAALARrwqL0Duf3QAB
// SIG // AAAAsDAJBgUrDgMCGgUAoIGqMBkGCSqGSIb3DQEJAzEM
// SIG // BgorBgEEAYI3AgEEMBwGCisGAQQBgjcCAQsxDjAMBgor
// SIG // BgEEAYI3AgEVMCMGCSqGSIb3DQEJBDEWBBTfyfVwxeYb
// SIG // z6DjJmGH7hlor5BI2zBKBgorBgEEAYI3AgEMMTwwOqAg
// SIG // gB4AVABGAFMALgBVAEkALgBkAGUAYgB1AGcALgBqAHOh
// SIG // FoAUaHR0cDovL21pY3Jvc29mdC5jb20wDQYJKoZIhvcN
// SIG // AQEBBQAEggEApjHePw9gt2oG3GfNpbAhR9gSo6vsmTLp
// SIG // NH7m3DMCA8+g+GF/jnEqEEQ8mawSOj3AfFh6pk6t5L07
// SIG // ul+75GE1giuckWYzhwvAgo7CdVHKVBajVExE8QxtFV1P
// SIG // VdKFkYd3gg6rQI7M3X3DktRB4rL5mI6Z0lBRVid8ppUz
// SIG // mRvxTSr0TF4s7FGYRrCUSh3jXaRWJovdRs0DrAlcd6xk
// SIG // eRxJhqq7YWPpbfXHhaUWXW1QmoY59oxqN/3fjcNH1NHz
// SIG // T7YGGYgjz8FxuPXaQSeUnyUf/2V8lV/+FWhZ6oteW/6N
// SIG // f8Gj9pH7InZCm0kOYIfhCZcAM0NiqOIcHPmzWXRg8bg5
// SIG // +aGCAigwggIkBgkqhkiG9w0BCQYxggIVMIICEQIBATCB
// SIG // jjB3MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGlu
// SIG // Z3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMV
// SIG // TWljcm9zb2Z0IENvcnBvcmF0aW9uMSEwHwYDVQQDExhN
// SIG // aWNyb3NvZnQgVGltZS1TdGFtcCBQQ0ECEzMAAAArOTJI
// SIG // wbLJSPMAAAAAACswCQYFKw4DAhoFAKBdMBgGCSqGSIb3
// SIG // DQEJAzELBgkqhkiG9w0BBwEwHAYJKoZIhvcNAQkFMQ8X
// SIG // DTEzMDMxNTA2MzM1NVowIwYJKoZIhvcNAQkEMRYEFL0T
// SIG // cFJOeOixb5AvK3LIX4sfbucMMA0GCSqGSIb3DQEBBQUA
// SIG // BIIBAHm7QbV5QJ7jAD5GsjSq2F15bkQvYDsEy3DUek1x
// SIG // +keCJZwdrgff4ib8GVt+/cvDB/1UcqrzIXLNjjszsdgn
// SIG // 4CtAqbERn0xFDRPq47tSH4uLE4zT55ieYLvB1dwnVksk
// SIG // qkl943sPaAhBHlRdXJVelhnujFRYXqV6YBh6RjPNBIxS
// SIG // 1dEkdZdQlUvi1fQ/bLKh/DkDo2sVk0wBMQSj4vgUgRi2
// SIG // fIXA25WD89tmcWrOoHfMJuZSDAmNWkCUQXCC3IJ44nv2
// SIG // KST/97gIlJxWroB7HA8wZ1hwurbI/Na2N/oVWmU8VLWq
// SIG // IIJYEInFUR9wV3a/zyy9pruZSFBQrUsNO5GQpL0=
// SIG // End signature block
